/*
 * Copyright (c) Doug Palmer <doug@charvolant.org> 2005
 *
 * See LICENSE for licensing details.
 * 
 * $Id$
 */

package org.charvolant.sudoku.gui;

import java.awt.Color;
import java.awt.Font;
import java.awt.MenuItem;
import java.awt.PopupMenu;
import java.io.*;

import org.charvolant.sudoku.Cell;
import org.charvolant.sudoku.Model;

/**
 * A list of preferences for how to display 
 * @author doug
 *
 */
public class DisplayPreferences extends Model implements Serializable {
  /** The default border colour. */
  public static final Color DEFAULT_BORDER = new Color(162, 162, 111);
  /** The default possible colour. */
  public static final Color DEFAULT_POSSIBLE = new Color(82, 255, 43);
  /** The default eliminated colour. */
  public static final Color DEFAULT_ELIMINATED = new Color(175, 175, 175);
  /** The default empty colour. */
  public static final Color DEFAULT_EMPTY = new Color(200, 31, 31);
  /** The default fixed colour. */
  public static final Color DEFAULT_SINGLETON = Color.BLUE;
  /** The default fixed colour. */
  public static final Color DEFAULT_FIXED = Color.BLACK;
  /** The default background colour. */
  public static final Color DEFAULT_BACKGROUND = new Color(255, 255, 174);
  /** The default fixed value display font. */
  public static final Font DEFAULT_FONT = new Font("SansSerif", Font.BOLD, 12);
  
  /** The value names; enough for a 5x5 grid */
  public static final String[] DEFAULT_NAMES = {
    "1", "2", "3", "4", "5", "6", "7", "8", "9",
    "A", "B", "C", "D", "E", "F", "G", "H", "I",
    "K", "L", "M", "N", "O", "P", "Q"
  };
  
  /** The background colour */
  private Color background;
  /** The border colour */
  private Color border;
  /** The possible colour */
  private Color possible;
  /** The eliminated colour */
  private Color eliminated;
  /** The empty colour */
  private Color empty;
  /** The singleton colout */
  private Color singleton;
  /** The fixed colour */
  private Color fixed;
  /** The display font */
  private Font font;
  /** The value names */
  private String[] names;
  
  public DisplayPreferences() {
    this.border = this.DEFAULT_BORDER;
    this.possible = this.DEFAULT_POSSIBLE;
    this.eliminated = this.DEFAULT_ELIMINATED;
    this.empty = this.DEFAULT_EMPTY;
    this.singleton = this.DEFAULT_SINGLETON;
    this.fixed = this.DEFAULT_FIXED;
    this.font = this.DEFAULT_FONT;
    this.background = this.DEFAULT_BACKGROUND;
    this.names = this.DEFAULT_NAMES;
  }

  /**
   * Get the background colour.
   *
   * @return Returns the background colour.
   */
  public Color getBackground() {
    return this.background;
  }

  /**
   * Set the background colour.
   *
   * @param background The background to set.
   */
  public void setBackground(Color background) {
    this.background = background;
    this.changed();
  }

  /**
   * Get the border colour.
   *
   * @return Returns the border colour.
   */
  public Color getBorder() {
    return this.border;
  }

  /**
   * Set the border colour.
   *
   * @param border The border to set.
   */
  public void setBorder(Color border) {
    this.border = border;
    this.changed();
  }

  /**
   * Get the eliminated colour.
   * <p>
   * This marks a possibility that has been eliminated
   * from consideration.
   *
   * @return Returns the eliminated colour.
   */
  public Color getEliminated() {
    return this.eliminated;
  }

  /**
   * Set the eliminated colour.
   *
   * @param eliminated The eliminated to set.
   */
  public void setEliminated(Color eliminated) {
    this.eliminated = eliminated;
    this.changed();
  }

  /**
   * Get the empty colour.
   * <p>
   * This colour represents a cell that has no possibilities.
   *
   * @return Returns the empty.
   */
  public Color getEmpty() {
    return this.empty;
  }

  /**
   * Set the empty colour.
   *
   * @param empty The colour to set.
   */
  public void setEmpty(Color empty) {
    this.empty = empty;
    this.changed();
  }

  /**
   * Get the fixed colour.
   * <p>
   * This is the colour of the text to use
   * when marking a fixed colour.
   *
   * @return Returns the fixed.
   */
  public Color getFixed() {
    return this.fixed;
  }

  /**
   * Set the fixed colour.
   *
   * @param fixed The fixed to set.
   */
  public void setFixed(Color fixed) {
    this.fixed = fixed;
    this.changed();
  }

  /**
   * Get the font.
   * <p>
   * This is the base font for displaying fixed values.
   *
   * @return Returns the font.
   */
  public Font getFont() {
    return this.font;
  }

  /**
   * Set the font.
   *
   * @param font The font to set.
   */
  public void setFont(Font font) {
    this.font = font;
    this.changed();
  }

  /**
   * Get the value names.
   *
   * @return Returns the names.
   */
  public String[] getNames() {
    return this.names;
  }

  /**
   * Set the value names.
   *
   * @param names The names to set.
   */
  public void setNames(String[] names) {
    this.names = names;
    this.changed();
  }

  /**
   * Get the possible colour.
   * <p>
   * This represents a value that is still possible.
   *
   * @return Returns the possible.
   */
  public Color getPossible() {
    return this.possible;
  }

  /**
   * Set the possible colour.
   *
   * @param possible The colour to set.
   */
  public void setPossible(Color possible) {
    this.possible = possible;
    this.changed();
  }

  /**
   * Get the singleton colour.
   * <p>
   * This marks a cell that has a singleton value.
   *
   * @return Returns the singleton.
   */
  public Color getSingleton() {
    return this.singleton;
  }

  /**
   * Set the singleton colour.
   *
   * @param singleton The colour to set.
   */
  public void setSingleton(Color singleton) {
    this.singleton = singleton;
    this.changed();
  }
  
  /**
   * Get the name of the value.
   * <p>
   * This goes into hexadecimal if the size is large enough.
   * 
   * @param v The value
   * 
   * @return A string representation of the value
   */
  public String getValueName(int v) {
    if (v < 0 || v >= this.names.length)
      return "?";
    return this.names[v];
  }
}
