/*
 * Copyright (c) Doug Palmer <doug@charvolant.org> 2005
 *
 * See LICENSE for licensing details.
 * 
 * $Id$
 */

package org.charvolant.sudoku;

import java.util.*;
/**
 * Something that has parameters which change.
 * <p>
 * Classes that implement {@link ModelListener} can register with
 * this class and be notified of changes.
 * 
 * @author doug
 * 
 * @see ModelListener
 * @see ModelEvent
 *
 */
public class Model {
  /** The list of listeners */
  private List<ModelListener> listeners;
  
  /**
   * Default constructor.
   */
  public Model() {
    this.listeners = null;
  }
  
  /**
   * Add a listener to the list of listeners.
   * 
   * @param listener The listener
   */
  public void addListener(ModelListener listener) {
    if (this.listeners == null)
      this.listeners = new ArrayList<ModelListener>();
    this.listeners.add(listener);
  }
  
  /**
   * Remove a listener from the list of listeners.
   * <p>
   * If a listener has been added more than once, it needs to be removed the
   * same number of times.
   * 
   * @param listener The listener to remove.
   */
  public void removeListener(ModelListener listener) {
    if (this.listeners == null)
      return;
    this.listeners.remove(listener);
    if (this.listeners.isEmpty())
      this.listeners = null;   
  }
  
  /**
   * Trigger a model update.
   * <p>
   * All listeners are informed of the change.
   */
  public void changed() {
    ModelEvent event = new ModelEvent(this);
    
    for (ModelListener listener: this.listeners)
      listener.modelUpdate(event);
  }
}
