/*
 * Copyright (c) Doug Palmer <doug@charvolant.org> 2005
 *
 * See LICENSE for licensing details.
 * 
 * $Id$
 */

package org.charvolant.sudoku;

/**
 * The context in which a solution search is done.
 * <p>
 * The context maintains a stack of choice-points, recording
 * choices made from alternatives.
 * The context also maintains a trail: a list of changes made
 * so that the changes can be unwound if a choice fails.
 * 
 * @author doug
 *
 */
public class Context {
  /** The list of choice points */
  private ChoicePoint[] choices;
  /** The choice point pointer */
  private int cp;
  /** The list of trail entries */
  private TrailEntry[] trail;
  /** The trail entry pointer */
  private int tp;
  
  /**
   * Default constructor.
   */
  public Context() {
    this.choices = new ChoicePoint[32];
    this.cp = 0;
    this.trail = new TrailEntry[256];
    this.tp = 0;
  }
  
  /**
   * Add a choice point.
   * <p>
   * The choices stack is automatically resized to fit.
   * 
   * @param entry The choice point entry
   */
  private void addChoicePoint(ChoicePoint entry) {
    while (this.cp >= this.choices.length) {
      ChoicePoint[] newChoices = new ChoicePoint[Math.max(this.cp, this.choices.length) * 2];
      for (int i = 0; i < this.choices.length; i++)
        newChoices[i] = this.choices[i];
      this.choices = newChoices;
    }
    this.choices[this.cp++] = entry;
  }
  
  /**
   * Add a choice point for a user-set fix
   */
  public void choiceUser() {
    this.addChoicePoint(new ChoicePoint(this.tp));
  }
  
  /**
   * Backtrack to a choice point.
   */
  public void backtrack() {
    ChoicePoint choice;
    
    if (this.cp <= 0)
      return;
    choice = this.choices[--this.cp];
    this.choices[this.cp] = null;
    while (this.tp > choice.getTrail()) {
      TrailEntry entry = this.trail[--this.tp];
      this.trail[this.tp] = null;
      entry.rewind();
    }
  }
  
  /**
   * Add a trail entry.
   * <p>
   * The trail is automatically resized to fit.
   * 
   * @param entry The trail entry
   */
  private void addTrail(TrailEntry entry) {
    while (this.tp >= this.trail.length) {
      TrailEntry[] newTrail = new TrailEntry[Math.max(this.tp, this.trail.length) * 2];
      for (int i = 0; i < this.trail.length; i++)
        newTrail[i] = this.trail[i];
      this.trail = newTrail;
    }
    this.trail[this.tp++] = entry;
  }
  
  /**
   * Trail the fixing of a cell.
   * 
   * @param cell The cell.
   */
  public void trailFix(Cell cell) {
    this.addTrail(new TrailFix(cell));
  }
  
  /**
   * Trail the a singleton cell.
   * 
   * @param cell The cell.
   */
  public void trailSingleton(Cell cell) {
    this.addTrail(new TrailSingleton(cell));
  }
  
  /**
   * Trail the elmination of a value from a cell.
   * 
   * @param cell The cell.
   * @param value The eliminated value
   */
  public void trailEliminate(Cell cell, int value) {
    this.addTrail(new TrailEliminate(cell, value));
  }
}
